<?php

namespace App\Livewire\Admin;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\Payment;
use App\Models\User;
use App\Models\Package;
use App\Models\Subscription;
use App\Models\Promotion;

class Payments extends Component
{
    use WithPagination;

    protected $paginationTheme = 'tailwind';

    public $showModal = false;

    public $user_id;
    public $package_id;
    public $amount;
    public $payment_method;
    public $original_amount;
    public $promotion_id = null;
    public $discount_percent = null;

    public function promotion()
{
    return $this->belongsTo(Promotion::class);
}


    public array $methods = [
        'efectivo',
        'tarjeta',
        'transferencia',
    ];

    protected function rules()
    {
        return [
            'user_id' => 'required|exists:users,id',
            'package_id' => 'required|exists:packages,id',
            'amount' => 'required|numeric|min:1',
            'payment_method' => 'required|in:efectivo,tarjeta,transferencia',
        ];
    }

    public function render()
    {
        return view('livewire.admin.payments', [
            'payments' => Payment::with(['user','package'])
                ->whereDate('created_at', now())
                ->orderBy('created_at', 'desc')
                ->paginate(10),

            'users' => User::role('cliente')->get(),
            'packages' => Package::where('is_active', true)->get(),
        ]);
    }

    public function updatedPackageId()
{
    $package = Package::find($this->package_id);

    if (! $package) {
        $this->amount = null;
        $this->original_amount = null;
        $this->promotion_id = null;
        $this->discount_percent = null;
        return;
    }

    // Precio base
    $this->original_amount = $package->price;
    $finalAmount = $package->price;

    // 🔥 Buscar promoción válida
    $promotion = Promotion::where('active', true)->get()
        ->first(fn ($promo) => $promo->isValid());

    if ($promotion) {
        $this->promotion_id = $promotion->id;
        $this->discount_percent = $promotion->discount_percent;

        // 👉 AQUÍ usamos tu método apply()
        $finalAmount = $promotion->apply($finalAmount);
    } else {
        $this->promotion_id = null;
        $this->discount_percent = null;
    }

    $this->amount = round($finalAmount, 2);
}


    public function create()
    {
        $this->resetFields();
        $this->showModal = true;
    }

    public function save()
    {
        $this->validate();

        $package = Package::findOrFail($this->package_id);

        /*
        |--------------------------------------------------------------------------
        | 1. REGISTRAR EL PAGO
        |--------------------------------------------------------------------------
        *//*
        $payment = Payment::create([
            'user_id' => $this->user_id,
            'package_id' => $package->id,
            'amount' => $this->amount,
            'payment_method' => $this->payment_method,
            'provider' => 'manual',
            'status' => 'pagado',
            'paid_at' => now(),
        ]);*/

        $payment = Payment::create([
            'user_id' => $this->user_id,
            'package_id' => $package->id,
            'subscription_id' => null, // se llena después
            'promotion_id' => $this->promotion_id,
            'promotion_percent' => $this->discount_percent,
            'amount' => $this->amount,
            'payment_method' => $this->payment_method,
            'provider' => 'manual',
            'status' => 'pagado',
            'paid_at' => now(),
        ]);


        /*
        |--------------------------------------------------------------------------
        | 2. CREAR SUSCRIPCIÓN AUTOMÁTICA
        |--------------------------------------------------------------------------
        */

        $startDate = now();
        $endDate = $startDate->copy()->addDays($package->duration_days);

        $subscription = Subscription::create([
            'user_id' => $this->user_id,
            'package_id' => $package->id,
            'start_date' => $startDate,
            'end_date' => $endDate,
            'status' => 'activa',
            'remaining_classes' => $package->classes_limit,
        ]);

        /*
        |--------------------------------------------------------------------------
        | 3. VINCULAR PAGO ↔ SUSCRIPCIÓN
        |--------------------------------------------------------------------------
        */
        $payment->update([
            'subscription_id' => $subscription->id,
        ]);

        $this->showModal = false;
        $this->resetFields();
    }


    private function resetFields()
    {
        $this->reset([
            'user_id',
            'package_id',
            'amount',
            'payment_method',
        ]);
    }
}
