<?php

namespace App\Livewire\Admin;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;

class Users extends Component
{
    use WithPagination;

    protected $paginationTheme = 'tailwind';

    /* UI */
    public $search = '';
    public $showModal = false;
    public $editMode = false;

    /* Form */
    public $userId;
    public $name = '';
    public $email = '';
    public $password = '';
    public $is_active = true;

    protected function rules()
    {
        return [
            'name' => ['required', 'string', 'min:3'],
            'email' => [
                'required',
                'email',
                Rule::unique('users', 'email')->ignore($this->userId),
            ],
            'password' => $this->editMode
                ? ['nullable', 'min:8']
                : ['required', 'min:8'],
        ];
    }

    public function render()
    {
        return view('livewire.admin.users', [
            'users' => User::role('cliente')
                ->where(function ($query) {
                    $query->where('name', 'like', "%{$this->search}%")
                          ->orWhere('email', 'like', "%{$this->search}%");
                })
                ->orderBy('name')
                ->paginate(10),
        ]);
    }

    /* =========================
        ACTIONS
    ========================= */

    public function create()
    {
        $this->resetFields();
        $this->editMode = false;
        $this->showModal = true;
    }

    public function edit(User $user)
    {
        $this->userId = $user->id;
        $this->name = $user->name;
        $this->email = $user->email;
        $this->is_active = (bool) $user->is_active;
        $this->password = '';

        $this->editMode = true;
        $this->showModal = true;
    }

    public function save()
    {
        $this->validate();

        if ($this->editMode) {

            $user = User::findOrFail($this->userId);

            $user->update([
                'name' => $this->name,
                'email' => $this->email,
                'is_active' => $this->is_active,
                'password' => $this->password
                    ? Hash::make($this->password)
                    : $user->password,
            ]);

        } else {

            $user = User::create([
                'name' => $this->name,
                'email' => $this->email,
                'password' => Hash::make($this->password),
                'is_active' => true,
            ]);

            // Rol FIJO: cliente
            $user->assignRole('cliente');
        }

        $this->showModal = false;
        $this->resetFields();
    }

    public function toggleStatus(User $user)
    {
        $user->update([
            'is_active' => ! $user->is_active
        ]);
    }

    /* =========================
        HELPERS
    ========================= */

    private function resetFields()
    {
        $this->reset([
            'userId',
            'name',
            'email',
            'password',
            'is_active',
        ]);
    }
}
