<?php

namespace App\Livewire\User;

use Livewire\Component;
use App\Models\ClassSession;
use App\Models\Attendance;
use App\Models\Subscription;
use Illuminate\Validation\ValidationException;

class ClassSchedule extends Component
{
  public function render()
{
  $userId = auth()->id();

  $subscription = \App\Models\Subscription::where('user_id', $userId)
      ->where('status', 'activa')
      ->first();

  $classes = \App\Models\ClassSession::with(['coach', 'discipline'])
      ->where('start_time', '>', now())
      ->whereIn('status', ['disponible', 'llena'])
      ->orderBy('start_time')
      ->get()
      ->map(function ($class) use ($userId) {

          // 🔥 CONSULTA DIRECTA (SIN RELACIONES)
          $class->is_enrolled = \App\Models\Attendance::where(
              'class_session_id', $class->id
          )->where(
              'user_id', $userId
          )->exists();

          $class->spots_left = max(
              $class->max_capacity -
              \App\Models\Attendance::where('class_session_id', $class->id)->count(),
              0
          );

          return $class;
      });

  return view('livewire.user.class-schedule', [
      'subscription' => $subscription,
      'classes' => $classes,
  ]);
}


    /**
     * AGENDAR CLASE
     */
    public function enroll(int $classId)
    {
        $user = auth()->user();

        // 1️⃣ Verificar suscripción activa
        $subscription = Subscription::where('user_id', $user->id)
            ->where('status', 'activa')
            ->first();

        if (! $subscription) {
            throw ValidationException::withMessages([
                'subscription' => 'No tienes una suscripción activa.',
            ]);
        }

        // 2️⃣ Verificar clases restantes
        if ($subscription->remaining_classes !== null && $subscription->remaining_classes <= 0) {
            throw ValidationException::withMessages([
                'classes' => 'No tienes clases disponibles.',
            ]);
        }

        $class = ClassSession::withCount('attendances')->findOrFail($classId);

        // 3️⃣ Verificar cupo
        if ($class->attendances_count >= $class->max_capacity) {
            throw ValidationException::withMessages([
                'capacity' => 'Esta clase ya está llena.',
            ]);
        }

        // 4️⃣ Verificar doble inscripción
        $alreadyEnrolled = Attendance::where('user_id', $user->id)
            ->where('class_session_id', $classId)
            ->exists();

        if ($alreadyEnrolled) {
            throw ValidationException::withMessages([
                'duplicate' => 'Ya estás inscrito en esta clase.',
            ]);
        }

        // 5️⃣ Crear asistencia
        Attendance::create([
            'user_id' => $user->id,
            'class_session_id' => $classId,
        ]);

        // 6️⃣ Descontar clase
        if ($subscription->remaining_classes !== null) {
            $subscription->decrement('remaining_classes');
        }

        $this->dispatch('notify', message: 'Clase agendada correctamente');
    }

    /**
     * CANCELAR CLASE
     */
    public function cancel(int $classId)
    {
        $user = auth()->user();

        $attendance = Attendance::where('user_id', $user->id)
            ->where('class_session_id', $classId)
            ->firstOrFail();

        $class = $attendance->classSession;

        // ❌ No permitir cancelar si faltan menos de X horas
        if (now()->diffInHours($class->start_time, false) < 3) {
            throw ValidationException::withMessages([
                'cancel' => 'No puedes cancelar una clase con menos de 3 horas de anticipación.',
            ]);
        }

        $attendance->delete();

        // ➕ Reponer clase
        $subscription = Subscription::where('user_id', $user->id)
            ->where('status', 'activa')
            ->first();

        if ($subscription && $subscription->remaining_classes !== null) {
            $subscription->increment('remaining_classes');
        }

        $this->dispatch('notify', message: 'Clase cancelada correctamente');
    }
}
